<?php
/**
 * Widget API: Newkarma_Tag_Widget class
 *
 * Author: Gian MR - http://www.gianmr.com
 *
 * @package Newkarma Core
 * @subpackage Widgets
 * @since 1.0.0
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Add the Tag widget.
 *
 * @since 1.0.0
 *
 * @see WP_Widget
 */
class Newkarma_Tag_Widget extends WP_Widget {
	/**
	 * Sets up a Recent Posts widget instance.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {
		$widget_ops = array(
			'classname'                   => 'newkarma-tag-cloud',
			'description'                 => __( 'A cloud of your most used tags.', 'newkarma' ),
			'customize_selective_refresh' => true,
		);
		parent::__construct( 'newkarma_tag_cloud', __( 'Tag Cloud(Newkarma Core)', 'newkarma' ), $widget_ops );
	}

	/**
	 * Outputs the content for the current Tag Cloud widget instance.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $args     Display arguments including 'before_title', 'after_title',
	 *                        'before_widget', and 'after_widget'.
	 * @param array $instance Settings for the current Tag Cloud widget instance.
	 */
	public function widget( $args, $instance ) {
		$title = apply_filters( 'widget_title', empty( $instance['title'] ) ? '' : $instance['title'], $instance, $this->id_base );

		echo $args['before_widget']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		if ( $title ) {
			echo $args['before_title'] . $title . $args['after_title']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}

		// Number tags.
		$newkar_number_tags = ( ! empty( $instance['newkar_number_tags'] ) ) ? absint( $instance['newkar_number_tags'] ) : absint( 5 );
		// Minimal Number Post.
		$number_posts = ( ! empty( $instance['number_posts'] ) ) ? absint( $instance['number_posts'] ) : absint( 1 );
		/* id tag */
		$idtag = ( ! empty( $instance['idtag'] ) ) ? array_map( 'absint', explode( ',', esc_attr( $instance['idtag'] ) ) ) : array( 0 );

		// Array for get_terms.
		$query_args = array(
			'orderby'      => 'count',
			'order'        => 'DESC',
			'hide_empty'   => true,
			'count'        => true,
			'hierarchical' => false,
		);

		if ( $newkar_number_tags ) {
			$query_args['number'] = $newkar_number_tags;
		}

		// if 'all tags' was selected ignore other selections of categories.
		if ( in_array( 0, $idtag, true ) ) {
			$idtag = array( 0 );
		}
		if ( ! in_array( 0, $idtag, true ) ) {
			$query_args['include'] = $idtag;
		}

		$tags = get_tags( $query_args );

		if ( ! is_wp_error( $tags ) && is_array( $tags ) && ! empty( $tags ) ) {
			echo '<div class="tagcloud"><ul class="newkarma-tag-lists">';
			foreach ( $tags as $tag ) {
				if ( $tag->count >= $number_posts ) {
					// Open item.
					echo '<li><a href="' . esc_url( get_term_link( (int) $tag->term_id ) ) . '" class="heading-text" title="' . esc_attr( $tag->name ) . '">';
					// Tag name.
					echo esc_attr( $tag->name );
					// Close item.
					echo '</a></li>';
				}
			}
			echo '</ul></div>';
		} else {
			echo '<div class="tagcloud">' . esc_html__( 'No Tags', 'newkarma' ) . '</div>';
		}
		echo $args['after_widget']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Handles updating settings for the current Tag Cloud widget instance.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $new_instance New settings for this instance as input by the user via
	 *                            WP_Widget::form().
	 * @param array $old_instance Old settings for this instance.
	 * @return array Settings to save or bool false to cancel saving.
	 */
	public function update( $new_instance, $old_instance ) {
		$instance = array(
			'title'              => '',
			'newkar_number_tags' => 5,
			'number_posts'       => 1,
			'idtag'              => '',
		);

		$instance['title'] = sanitize_text_field( $new_instance['title'] );
		// Number tag.
		$instance['newkar_number_tags'] = absint( $new_instance['newkar_number_tags'] );
		// Number posts.
		$instance['number_posts'] = absint( $new_instance['number_posts'] );
		// tag id.
		$instance['idtag'] = wp_strip_all_tags( $new_instance['idtag'] );

		return $instance;
	}

	/**
	 * Outputs the Tag Cloud widget settings form.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $instance Current settings.
	 */
	public function form( $instance ) {
		$instance = wp_parse_args(
			(array) $instance,
			array(
				'title'              => __( 'Tag Populer', 'newkarma' ),
				'number_posts'       => 1,
				'newkar_number_tags' => 5,
				'idtag'              => '',
			)
		);

		$title = sanitize_text_field( $instance['title'] );
		// Number tags.
		$newkar_number_tags = absint( $instance['newkar_number_tags'] );
		// Number posts.
		$number_posts = absint( $instance['number_posts'] );
		// tag id.
		$idtag = wp_strip_all_tags( $instance['idtag'] );

		echo '<p><label for="' . esc_html( $title ) . '">' . esc_html__( 'Title:', 'newkarma' ) . '</label>
			<input type="text" class="widefat" id="' . esc_html( $title ) . '" name="' . esc_html( $this->get_field_name( 'title', 'newkarma' ) ) . '" value="' . esc_html( $instance['title'] ) . '" />
		</p>';

		echo '<p>
			<label for="' . esc_attr( $this->get_field_id( 'number_posts' ) ) . '">' . esc_html__( 'Number post', 'newkarma' ) . '</label>
			<input class="widefat" id="' . esc_attr( $this->get_field_id( 'number_posts' ) ) . '" name="' . esc_attr( $this->get_field_name( 'number_posts' ) ) . '" type="number" value="' . esc_attr( $number_posts ) . '" />
			<br />
			<small>' . esc_html__( 'Minimal Posts For Displaying Tags. For example you want display tag if the tag have 6 posts, you can insert 6. Default is 1', 'newkarma' ) . '</small>		
		</p>';

		echo '<p>
			<label for="' . esc_html( $this->get_field_id( 'newkar_number_tags' ) ) . '">' . esc_html__( 'Number post', 'newkarma' ) . '</label>
			<input class="widefat" id="' . esc_html( $this->get_field_id( 'newkar_number_tags' ) ) . '" name="' . esc_html( $this->get_field_name( 'newkar_number_tags' ) ) . '" type="number" value="' . esc_attr( $newkar_number_tags ) . '" />
		</p>';

		echo '<p>
			<label for="' . esc_attr( $this->get_field_id( 'idtag' ) ) . '">' . esc_html__( 'Tag ID:', 'newkarma' ) . '</label>
			<input class="widefat" id="' . esc_attr( $this->get_field_id( 'idtag' ) ) . '" name="' . esc_attr( $this->get_field_name( 'idtag' ) ) . '" type="text" value="' . esc_attr( $idtag ) . '" />
			<br />
			<small>' . esc_html__( 'You can include exclusive tags, just insert ID Tags separate by commas(,). Example: 1,2,3,4. Leave blank if you want display by default tags', 'newkarma' ) . '</small>
		</p>';
	}
}

add_action(
	'widgets_init',
	function() {
		register_widget( 'Newkarma_Tag_Widget' );
	}
);
