<?php
// Simple File Manager - Full Featured
error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();

// Get current directory safely
function getCurrentPath($dir = null) {
    if ($dir && is_dir($dir)) {
        return realpath($dir);
    }
    return realpath(getcwd());
}

$current_dir = getCurrentPath($_GET['dir'] ?? null);
$message = '';

// Handle POST actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // File Upload
    if (isset($_FILES['file_upload']) && $_FILES['file_upload']['error'] === 0) {
        $filename = basename($_FILES['file_upload']['name']);
        $target_path = $current_dir . '/' . $filename;
        if (move_uploaded_file($_FILES['file_upload']['tmp_name'], $target_path)) {
            $message = "✅ File uploaded: " . htmlspecialchars($filename);
        } else {
            $message = "❌ Upload failed";
        }
    }
    
    // Create Folder
    if (!empty($_POST['new_folder'])) {
        $folder_name = trim($_POST['new_folder']);
        if ($folder_name) {
            $folder_path = $current_dir . '/' . $folder_name;
            if (!is_dir($folder_path)) {
                if (mkdir($folder_path, 0755)) {
                    $message = "✅ Folder created: " . htmlspecialchars($folder_name);
                } else {
                    $message = "❌ Failed to create folder";
                }
            } else {
                $message = "❌ Folder already exists";
            }
        }
    }
    
    // Create File
    if (!empty($_POST['new_file'])) {
        $file_name = trim($_POST['new_file']);
        if ($file_name) {
            $file_path = $current_dir . '/' . $file_name;
            if (!file_exists($file_path)) {
                if (file_put_contents($file_path, '') !== false) {
                    $message = "✅ File created: " . htmlspecialchars($file_name);
                } else {
                    $message = "❌ Failed to create file";
                }
            } else {
                $message = "❌ File already exists";
            }
        }
    }
    
    // Save file content
    if (isset($_POST['file_content']) && isset($_POST['file_path'])) {
        if (file_put_contents($_POST['file_path'], $_POST['file_content']) !== false) {
            $message = "✅ File saved successfully";
        } else {
            $message = "❌ Failed to save file";
        }
    }
}

// Handle GET actions
if (isset($_GET['action']) && isset($_GET['file'])) {
    $file_path = $current_dir . '/' . basename($_GET['file']);
    
    if (file_exists($file_path)) {
        switch ($_GET['action']) {
            case 'delete':
                if (unlink($file_path)) {
                    $message = "✅ File deleted: " . htmlspecialchars($_GET['file']);
                } else {
                    $message = "❌ Failed to delete file";
                }
                break;
                
            case 'edit':
                // Editing handled in the form below
                break;
                
            case 'rename':
                if (isset($_POST['new_name'])) {
                    $new_name = trim($_POST['new_name']);
                    if ($new_name) {
                        $new_path = $current_dir . '/' . $new_name;
                        if (rename($file_path, $new_path)) {
                            $message = "✅ File renamed to: " . htmlspecialchars($new_name);
                        } else {
                            $message = "❌ Failed to rename file";
                        }
                    }
                }
                break;
        }
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>File Manager</title>
    <style>
        body { 
            font-family: Arial, sans-serif; 
            margin: 20px; 
            background: #f5f5f5; 
            line-height: 1.6;
        }
        .container { 
            background: white; 
            padding: 20px; 
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .message { 
            padding: 12px; 
            background: #d4edda; 
            color: #155724; 
            margin: 15px 0; 
            border-radius: 4px;
            border: 1px solid #c3e6cb;
        }
        .message.error { 
            background: #f8d7da; 
            color: #721c24;
            border-color: #f5c6cb;
        }
        .file-list { 
            margin: 25px 0; 
            border: 1px solid #ddd;
            border-radius: 5px;
            overflow: hidden;
        }
        .file-item { 
            padding: 12px 15px; 
            border-bottom: 1px solid #eee; 
            display: flex;
            align-items: center;
        }
        .file-item:hover {
            background: #f8f9fa;
        }
        .file-item:last-child {
            border-bottom: none;
        }
        .actions { 
            margin-left: auto;
            display: flex;
            gap: 8px;
        }
        .action-btn { 
            padding: 4px 12px; 
            text-decoration: none; 
            border-radius: 4px; 
            font-size: 12px; 
            border: none;
            cursor: pointer;
        }
        .edit { background: #28a745; color: white; }
        .delete { background: #dc3545; color: white; }
        .rename { background: #ffc107; color: black; }
        .nav-btn {
            display: inline-block;
            padding: 8px 16px;
            background: #007bff;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            margin: 5px 0;
        }
        .section {
            margin: 25px 0;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 6px;
        }
        .section h3 {
            margin-top: 0;
            color: #333;
        }
        input[type="text"], input[type="file"] {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            width: 300px;
            margin: 5px 0;
        }
        button, .btn {
            padding: 8px 16px;
            background: #007bff;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }
        button:hover, .btn:hover {
            background: #0056b3;
        }
        textarea {
            width: 100%;
            height: 400px;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-family: monospace;
            font-size: 14px;
            margin: 10px 0;
        }
        .form-group {
            margin: 15px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>📁 File Manager</h1>
        
        <?php if ($message): ?>
            <div class="message <?= strpos($message, '❌') !== false ? 'error' : '' ?>">
                <?= htmlspecialchars($message) ?>
            </div>
        <?php endif; ?>

        <p><strong>Current Directory:</strong> <?= htmlspecialchars($current_dir) ?></p>
        
        <div class="navigation">
            <a href="?dir=<?= urlencode(dirname($current_dir)) ?>" class="nav-btn">↑ Parent Directory</a>
        </div>

        <!-- File Listing -->
        <div class="section">
            <h3>📂 Files and Folders</h3>
            <div class="file-list">
                <?php
                $items = scandir($current_dir);
                if ($items) {
                    foreach ($items as $item) {
                        if ($item == '.' || $item == '..') continue;
                        
                        $full_path = $current_dir . '/' . $item;
                        $is_dir = is_dir($full_path);
                        
                        echo '<div class="file-item">';
                        if ($is_dir) {
                            echo '📁 <a href="?dir=' . urlencode($full_path) . '" style="font-weight: bold;">' . htmlspecialchars($item) . '</a>';
                        } else {
                            echo '📄 ' . htmlspecialchars($item);
                            echo '<span class="actions">';
                            echo '<a href="?dir=' . urlencode($current_dir) . '&action=edit&file=' . urlencode($item) . '" class="action-btn edit">Edit</a>';
                            echo '<a href="?dir=' . urlencode($current_dir) . '&action=rename&file=' . urlencode($item) . '" class="action-btn rename">Rename</a>';
                            echo '<a href="?dir=' . urlencode($current_dir) . '&action=delete&file=' . urlencode($item) . '" class="action-btn delete" onclick="return confirm(\'Delete \\\'' . htmlspecialchars($item) . '\\\'?\')">Delete</a>';
                            echo '</span>';
                        }
                        echo '</div>';
                    }
                } else {
                    echo '<p>Unable to read directory</p>';
                }
                ?>
            </div>
        </div>

        <!-- File Editor -->
        <?php if (isset($_GET['action']) && $_GET['action'] == 'edit' && isset($_GET['file'])): ?>
            <?php
            $file_path = $current_dir . '/' . basename($_GET['file']);
            $file_content = file_exists($file_path) ? file_get_contents($file_path) : '';
            ?>
            <div class="section">
                <h3>✏️ Editing: <?= htmlspecialchars($_GET['file']) ?></h3>
                <form method="POST">
                    <input type="hidden" name="file_path" value="<?= htmlspecialchars($file_path) ?>">
                    <textarea name="file_content" placeholder="File content..."><?= htmlspecialchars($file_content) ?></textarea>
                    <div class="form-group">
                        <button type="submit">💾 Save File</button>
                        <a href="?dir=<?= urlencode($current_dir) ?>" class="btn" style="background: #6c757d;">❌ Cancel</a>
                    </div>
                </form>
            </div>

        <!-- Rename Form -->
        <?php elseif (isset($_GET['action']) && $_GET['action'] == 'rename' && isset($_GET['file'])): ?>
            <div class="section">
                <h3>✏️ Rename: <?= htmlspecialchars($_GET['file']) ?></h3>
                <form method="POST">
                    <div class="form-group">
                        <input type="text" name="new_name" value="<?= htmlspecialchars($_GET['file']) ?>" required>
                    </div>
                    <button type="submit">📝 Rename</button>
                    <a href="?dir=<?= urlencode($current_dir) ?>" class="btn" style="background: #6c757d;">❌ Cancel</a>
                </form>
            </div>

        <?php else: ?>
            <!-- Upload Form -->
            <div class="section">
                <h3>📤 Upload File</h3>
                <form method="POST" enctype="multipart/form-data">
                    <div class="form-group">
                        <input type="file" name="file_upload" required>
                    </div>
                    <button type="submit">🚀 Upload File</button>
                </form>
            </div>

            <!-- Create Folder -->
            <div class="section">
                <h3>📁 Create New Folder</h3>
                <form method="POST">
                    <div class="form-group">
                        <input type="text" name="new_folder" placeholder="Enter folder name" required>
                    </div>
                    <button type="submit">✅ Create Folder</button>
                </form>
            </div>

            <!-- Create File -->
            <div class="section">
                <h3>📄 Create New File</h3>
                <form method="POST">
                    <div class="form-group">
                        <input type="text" name="new_file" placeholder="Enter file name" required>
                    </div>
                    <button type="submit">✅ Create File</button>
                </form>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>